defmodule Reseau.Etape.Plateau do
  @moduledoc """
  Une étape pour jouer dans la partie.
  """

  @behaviour Reseau.Etape

  @typedoc "Un joueur"
  @type joueur() :: Reseau.Joueur.t()

  @typedoc "Une partie"
  @type partie() :: Reseau.Partie.t()

  @impl true
  @doc """
  Callback appelée quand l'étape est chargée et prête à être affichée.

  Arguments :

  * `joueur` : le joueur actuel (`Reseau.Joueur`) ;
  * `partie` : la partie actuelle (`Reseau.Partie`).

  Elle doit retourner un tuple contenant le joueur et la partie (éventuellement modifiés).

  """
  @spec entrer(joueur(), partie()) :: {joueur(), partie()}
  def entrer(joueur, partie), do: {joueur, partie}

  @impl true
  @doc """
  Retourne le titre de l'étape.
  """
  @spec titre(joueur(), partie()) :: String.t()
  def titre(_joueur, partie), do: partie.titre

  @impl true
  @doc """
  Retourne le texte de l'étape.
  """
  @spec texte(joueur(), partie()) :: String.t()
  def texte(joueur, partie) do
    coups_valides = coups_valides(joueur, partie)
    titres =
      Enum.map(coups(joueur, partie), fn {coup, affichage, _} ->
        index = Enum.find_index(coups_valides, fn {valeur, _, _} -> coup == valeur end)

        if index do
          "[#{index + 1}] - #{affichage}"
        else
          "[-] - #{affichage}"
        end
      end)

    """
    Entrez le numéro d'une carte ou d'un coup pour le jouer dans cette partie.
    Les coups précédés de [-] ne sont pas actuellement valides.

    #{Enum.join(titres, "\n")}
    """
  end

  @impl true
  @doc """
  Retourne le prompt de l'étape.
  """
  @spec prompt(joueur(), partie()) :: String.t()
  def prompt(_joueur, _partie), do: "Entrez le numéro du coup à jouer :"

  @impl true
  @doc """
  Gère les commandes entrées par l'utilisateur.

  La commande entrée est précisée sous la forme d'une chaîne de caractères. Le retour peut être de plusieurs types :

  - `:silence` : n'affiche rien et ne fait rien ;
  - `:prompt` : n'affiche rien sauf le prompt ;
  - `:rafraîchir` : n'affiche que l'écran de nouveau, sans message ;
  - `{:silence, message}` : affiche juste le message ;
  - `{:prompt, message}` : n'affiche que le message suivi du prompt ;
  - `{:rafraîchir, message}` : affiche le message suivi de tout l'écran ;
  - `{:rediriger, module, joueur, partie}` : change l'étape actuelle, affiche le nouvel écran ;
  - `{:rediriger, module, joueur, partie, message}` : change l'étape actuelle, affiche le message suivi du nouvel écran ;
  - `{:rediriger, module, joueur, partie, :silence}` : change l'étape actuelle, n'affiche rien ;
  - `{:rediriger, module, joueur, partie, :silence, message}` : change l'étape actuelle, affiche uniquement le message.

  """
  @spec gérer_commandes(joueur(), partie(), String.t()) ::
              :silence
              | :prompt
              | :rafrîchir
              | {:silence, String.t()}
              | {:prompt, String.t()}
              | {:rafraîchir, String.t()}
              | {:rediriger, module(), joueur(), partie()}
              | {:rediriger, module(), joueur(), partie(), String.t()}
              | {:rediriger, module(), joueur(), partie(), :silence}
              | {:rediriger, module(), joueur(), partie(), :silence, String.t()}
  def gérer_commandes(joueur, partie, commande) do
    with {index, _} <- Integer.parse(commande),
         {:ok, coup} <- Enum.fetch(coups_valides(joueur, partie), index - 1) do
      {coup, affichage, _} = coup
      Jeu.jouer(partie.id, partie.processus, joueur.id, coup)
      {:silence, affichage}
    else
      :error -> {:prompt, "Ce coup n'est pas valide."}
    end
  end

  defp coups(joueur, partie) do
    Partie.coups(partie.processus, joueur.id)
  end

  defp coups_valides(joueur, partie) do
    Enum.filter(coups(joueur, partie), fn {_, _, peut_jouer?} -> peut_jouer? end)
  end
end
